const { Op } = require("sequelize");
const Journey = require("../models/Journey");
const Memorial = require("../models/Memorial");
const User = require("../models/User");
const CoAdmins = require("../models/CoAdmins");

module.exports = {
  async index(req, res) {
    try {
      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      const journeys = await Journey.findAndCountAll({
        where: {
          ...(req.query.search && {
            [Op.or]: [
              { title: { [Op.regexp]: req.query.search } },
              { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
              { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
              { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
            ],
          }),
        },
        order: [["createdAt", "DESC"]],
        limit: parseInt(pageSize),
        offset: (parseInt(page) - 1) * parseInt(pageSize),
        include: [{ model: Memorial, as: "memorial" }],
      });
      const total = Math.ceil(journeys.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        journeys: journeys.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

  async getJourneysByCoAdmins(req, res) {
    try {
      // Step 1: Get all memorials where the user is a co-admin
      const coAdminMemorials = await CoAdmins.findAll({
        where: { user_id: req.params.userId },
        attributes: ["memorial_id"],
      });
      // Extract memorial IDs from the result
      const memorialIds = coAdminMemorials.map(
        (coAdmin) => coAdmin.memorial_id
      );
      console.log({ memorialIds });

      if (memorialIds.length === 0) {
        return []; // No memorials found for this co-admin
      }

      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      // Step 2: Fetch journeys linked to these memorial IDs
      const journeys = await Journey.findAndCountAll({
        where: { memorial_id: memorialIds },
        ...(req.query.search && {
          [Op.or]: [
            { title: { [Op.regexp]: req.query.search } },
            { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
            { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
            { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
          ],
        }),
        include: [
          {
            model: Memorial,
            as: "memorial",
          },
        ],
      });
      const total = Math.ceil(journeys.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        journeys: journeys.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

 async create(req, res) {
  try {
    const body = { ...req.body };

    // yahan pe tum frontend se bhej rahe ho
    const loggedInUserId = body.user_id;

    // Memorial fetch karo
    const memorial = await Memorial.findByPk(body.memorial_id);

    if (!memorial) {
      return res.status(404).json({ message: "Memorial not found" });
    }

    // Check karo ke memorial ka creator hi journey post kar raha hai
    if (memorial.user_id !== parseInt(loggedInUserId)) {
      return res.status(403).json({
        message: "Only the memorial creator can post on journey of life",
      });
    }

    // Agar file hai to image add karo
    if (req.file) {
      body["image"] = req.file.path;
    }

    // Journey create karo
    const journey = await Journey.create({
      ...body,
    });

    return res.json(journey);

  } catch (err) {
    console.log({ err });
    return res.status(500).json({ message: "Server error" });
  }
}
,

  async getJourney(req, res) {
    try {
      const journey = await Journey.findByPk(req.params.id, {
        include: [{ model: User, as: "user" }],
      });
      return res.json(journey);
    } catch (err) {
      console.log({ err });
    }
  },

  async getJourneyByMemorial(req, res) {
    try {
      const journeys = await Journey.findAll({
        where: { memorial_id: req.params.memorialId },
        include: [{ model: User, as: "user" }],
        order: [["createdAt", "DESC"]],
      });
      if (!journeys.length)
        return res
          .status(400)
          .json({ message: "There are no journeys with that memorial Id" });
      return res.json(journeys);
    } catch (err) {
      console.log({ err });
    }
  },

  async updateJourney(req, res) {
    try {
      console.log({ body: req.body });
      const journey = await Journey.findByPk(req.params.id);
      journey.title = req.body.title;
      journey.description = req.body.description;
      if (req.file) {
        journey.image = req.file.path;
      }
      await journey.save();
      return res.json({ message: "Updated journey successfully!" });
    } catch (err) {
      console.log({ err });
    }
  },

  async removeJourney(req, res) {
    try {
      await Journey.destroy({ where: { id: req.params.id } });
      return res.json({ message: "Removed journey successfully!" });
    } catch (err) {
      console.log({ err });
    }
  },
};
