const { Op } = require("sequelize");
const Memorial = require("../models/Memorial");
const User = require("../models/User");
const CoAdmins = require("../models/CoAdmins");
const MemorialAudioBG = require("../models/MemorialAudioBG");
const Notification = require("../models/Notification");

const isInvalidDate = (v) =>
  !v || v === "0000-00-00" || Number.isNaN(new Date(v).getTime());

const toDate = (v) => new Date(v);

const toDateOnlyUTC = (date) =>
  new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate()));

const addYearsUTC = (date, years) => {
  const d = new Date(date);
  d.setUTCFullYear(d.getUTCFullYear() + years);
  return d;
};

const index = async (req, res) => {
  try {
    const memorials = await Memorial.findAll();
    return res.json(memorials);
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const getAllMemorialAdmin = async (req, res) => {
  try {
    const page = req.query.page ? parseInt(req.query.page) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize) : 4;

    // ✅ Get all users with type = 'admin'
    const admins = await User.findAll({
      where: { type: "admin" },
      attributes: ["userid"], // only need their ids
    });

    const adminIds = admins.map((u) => u.userid);

    if (adminIds.length === 0) {
      return res.status(200).json({ filters: { page, total: 0 }, memorials: [] });
    }

    let whereCondition = {
      user_id: { [Op.in]: adminIds }, // ✅ get only admin memorials
    };

    // ✅ Optional search
    if (req.query.search) {
      whereCondition = {
        ...whereCondition,
        [Op.or]: [
          { Firstname: { [Op.regexp]: req.query.search } },
          { MiddleName: { [Op.regexp]: req.query.search } },
          { LastName: { [Op.regexp]: req.query.search } },
        ],
      };
    }

    // ✅ Fetch memorials
    const memorials = await Memorial.findAndCountAll({
      where: whereCondition,
      order: [["createdOn", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
    });

    const total = Math.ceil(memorials.count / pageSize);

    return res.status(200).json({
      filters: { page, nextPage: page + 1, total },
      memorials: memorials.rows,
    });
  } catch (error) {
    console.error("❌ getAllMemorialAdmin error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const getAllMemorial = async (req, res) => {
  try {
    const page = req.query.page ? req.query.page : 1;
    const pageSize = req.query.pageSize ? req.query.pageSize : 4;
    const memorials = await Memorial.findAndCountAll({
      where: {
        
        ...(req.query.search && {
          [Op.or]: [
            { Firstname: { [Op.regexp]: req.query.search } },
            { MiddleName: { [Op.regexp]: req.query.search } },
            { LastName: { [Op.regexp]: req.query.search } },
          ],
        }),
      },
      
      order: [["createdOn", "DESC"]],
      limit: parseInt(pageSize),
      offset: (parseInt(page) - 1) * parseInt(pageSize),
    });
    const total = Math.ceil(memorials.count / pageSize);
    return res.status(201).json({
      filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
      memorials: memorials.rows,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const getPinnedMemorials = async (req, res) => {
  try {
    const memorials = await Memorial.findAll({
      where: { pinned: true },
    });
    return res.json(memorials);
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const getByCategory = async (req, res) => {
  try {
    const page = req.query.page ? req.query.page : 1;
    const pageSize = req.query.pageSize ? req.query.pageSize : 4;
    const memorials = await Memorial.findAndCountAll({
      where: {
        suspended: false,
        category_id: req.params.categoryId,
        ...(req.query.search && {
          [Op.or]: [
            { Firstname: { [Op.regexp]: req.query.search } },
            { MiddleName: { [Op.regexp]: req.query.search } },
            { LastName: { [Op.regexp]: req.query.search } },
          ],
        }),
      },
      order: [["createdOn", "DESC"]],
      limit: parseInt(pageSize),
      offset: (parseInt(page) - 1) * parseInt(pageSize),
    });
    const total = Math.ceil(memorials.count / pageSize);
    return res.status(201).json({
      filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
      memorials: memorials.rows,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const createMemorial = async (req, res) => {
  try {
    req.body["CreatedOn"] = new Date();

    if (!req.body.lastRenewed) {
      req.body.lastRenewed = new Date();
    }

    const lastRenewedDate = new Date(req.body.lastRenewed);
    const nextRenewalDate = new Date(lastRenewedDate);
    nextRenewalDate.setFullYear(nextRenewalDate.getFullYear() + 3);
    req.body["nextRenewal"] = nextRenewalDate;

    // helper to normalize and strip public/
    const cleanPath = (filePath) =>
      filePath.replace(/\\/g, "/").replace(/^public\//, "");

    // handle optional uploaded files
    if (req.coverimage) {
      req.body.coverPhoto = req.coverimage;
    }
    if (req.files?.coverPhoto) {
      req.body.coverPhoto = cleanPath(req.files.coverPhoto[0].path);
    }
    if (req.files?.ProfileImgUrl) {
      req.body.ProfileImgUrl = cleanPath(req.files.ProfileImgUrl[0].path);
    }
    if (req.files?.audio) {
      req.body.audio = cleanPath(req.files.audio[0].path);
    }

    const newMemorial = await Memorial.create(req.body);

    // assign co-admin
    const coAdmin = await CoAdmins.create({
      memorial_id: newMemorial.id,
      user_id: req.body.user_id,
    });

    // create notification
    await Notification.create({
      text: "A memorial is being created",
      user_id: req.body.user_id,
      type: "memorial",
    });

    return res.status(201).json({
      message: "Memorial created successfully",
      memorial: newMemorial,
      coAdmin,
    });
  } catch (error) {
    console.error("❌ createMemorial error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};



const updateProfileImageMemorial = async (req, res) => {
  try {
    const { memorialId } = req.body;

    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    let filePath = req.file.path;
    filePath = filePath.replace("public", "api"); // Ajuste se usar "public" como raiz
    const imagePath = filePath;

    const memorial = await Memorial.update(
      { ProfileImgUrl: imagePath },
      {
        where: { id: memorialId },
      }
    );

    return res.status(200).json({ message: "Upload successful", memorial, filePath: imagePath });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const updateCoverPhotoMemorial = async (req, res) => {
  try {
    const { memorialId } = req.body;

    // Ensure a file was uploaded
    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    // Adjust the path if needed (same as profile image)
    let filePath = req.file.path;
    filePath = filePath.replace("public", "api");
    req.file.path = filePath;
    const coverPhotoPath = req.file.path;

    // Update the memorial record
    const memorial = await Memorial.update(
      { coverPhoto: coverPhotoPath },
      {
        where: {
          id: memorialId,
        },
      }
    );

    return res.status(200).json({ message: "Cover photo uploaded successfully", memorial });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const updateCover = async (req, res) => {
  try {
    const { memorialId } = req.params;
    let filePath = req?.file?.path;
    filePath = filePath.replace("public", "api");
    req.file.path = filePath;
    const imagePath = req.file.path;
    const memorial = await Memorial.update(
      { coverPhoto: imagePath },
      {
        where: {
          id: memorialId,
        },
      }
    );

    return res.status(200).json({ message: "Upload successful", memorial });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const pinMemorial = async (req, res) => {
  try {
    const memorial = await Memorial.findByPk(req.params.id);
    memorial.pinned = true;
    await memorial.save();
    return res.json({
      message: "Pinned the memorial",
      pinned: memorial.pinned,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};
const unpinMemorial = async (req, res) => {
  try {
    const memorial = await Memorial.findByPk(req.params.id);
    memorial.pinned = false;
    await memorial.save();
    return res.json({
      message: "Unpinned the memorial",
      pinned: memorial.pinned,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const deleteMemorial = async (req, res) => {
  try {
    const { memorialId } = req.params;
    const coAdmins = await CoAdmins.findAll({
      where: { memorial_id: memorialId },
    });
    if (coAdmins.length > 0) {
      await CoAdmins.destroy({ where: { memorial_id: memorialId } });
    }

    const foundBGAudio = await MemorialAudioBG.findOne({
      where: { memorial_id: memorialId },
    });
    if (foundBGAudio) {
      await MemorialAudioBG.destroy({ where: { memorial_id: memorialId } });
    }

    await Memorial.destroy({ where: { id: memorialId } });

    return res.json({ message: "Memorial deleted successfully" });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const getMemorialsByUserId = async (req, res) => {
  const { userId } = req.params;
  console.log({ userId });
  try {
    const memorials = await Memorial.findAll(
      { where: { user_id: userId } },
      { include: User }
    );
    // console.log({ memorials: memorials.getUser() });
    return res.status(200).json(memorials);
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const getMemorialById = async (req, res) => {
  const { id } = req.params;
  try {
    // Find the memorial by its ID
    const memorial = await Memorial.findByPk(id);
    console.log("data memorial" , memorial);
    if (!memorial) {
      return res.status(404).json({ error: "Memorial not found" });
    }
    res.status(200).json(memorial);
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const getMemorialsByAddress = async (req, res) => {
  const { address } = req.params;
  try {
    // Find the memorial by its ID
    const memorial = await Memorial.findOne({
      where: { MemorialAddress: address },
    });
    if (!memorial) {
      return res.status(404).json({ error: "Memorial not found" });
    }
    res.status(200).json(memorial);
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const updateMemorial = async (req, res) => {
  try {
    const { memorialId } = req.params;

    // Clone body so we can safely mutate it
    const updateData = { ...req.body };

    // Normalize helper to clean "public/" prefix
    const cleanPath = (filePath) =>
      filePath.replace(/\\/g, "/").replace(/^public\//, "");

    // Handle uploaded files
    if (req.files?.coverPhoto) {
      updateData.coverPhoto = cleanPath(req.files.coverPhoto[0].path);
    }
    if (req.files?.ProfileImgUrl) {
      updateData.ProfileImgUrl = cleanPath(req.files.ProfileImgUrl[0].path);
    }
    if (req.files?.audio) {
      updateData.audio = cleanPath(req.files.audio[0].path);
    }

    // Perform update
    await Memorial.update(updateData, { where: { id: memorialId } });

    // Fetch updated memorial
    const memorial = await Memorial.findByPk(memorialId);

    return res.status(200).json({
      message: "Memorial updated successfully",
      memorial,
    });
  } catch (error) {
    console.error("Error updating memorial:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};



const suspendMemorial = async (req, res) => {
  try {
    const memorial = await Memorial.findOne({ where: { id: req.params.id } });
    memorial.suspended = true;
    await memorial.save();
    return res.json({
      message: "Suspended the memorial successfully!",
      suspended: memorial.suspended,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const unsuspendMemorial = async (req, res) => {
  try {
    const memorial = await Memorial.findOne({ where: { id: req.params.id } });
    memorial.suspended = false;
    await memorial.save();
    return res.json({
      message: "Unsuspended the memorial successfully!",
      suspended: memorial.suspended,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const addView = async (req, res) => {
  try {
    const memorial = await Memorial.findOne({ where: { id: req.params.id } });
    memorial.views++;
    await memorial.save();
    return res.json(memorial.views);
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};


const updateMemorialRenewal = async (req, res) => {
  try {
    const id = req.params.id;
    const body = req.body || {};
    const now = toDateOnlyUTC(new Date());

    const memorial = await Memorial.findByPk(id);
    if (!memorial) {
      return res.status(404).json({ message: "Memorial not found" });
    }

    // Optional authorization example
    // if (req.user.role !== 'admin' && req.user.userid !== memorial.user_id) {
    //   return res.status(403).json({ message: "Forbidden" });
    // }

    // Resolve lastRenewed
    let lastRenewed;
    if (!isInvalidDate(body.lastRenewed)) {
      lastRenewed = toDateOnlyUTC(toDate(body.lastRenewed));
    } else if (!isInvalidDate(memorial.lastRenewed)) {
      lastRenewed = toDateOnlyUTC(toDate(memorial.lastRenewed));
    } else if (!isInvalidDate(memorial.CreatedOn)) {
      lastRenewed = toDateOnlyUTC(toDate(memorial.CreatedOn));
    } else {
      lastRenewed = now;
    }

    // Resolve nextRenewal
    let nextRenewal = null;
    if (!isInvalidDate(body.nextRenewal)) {
      nextRenewal = toDateOnlyUTC(toDate(body.nextRenewal));
      if (nextRenewal.getTime() <= lastRenewed.getTime()) {
        nextRenewal = toDateOnlyUTC(addYearsUTC(lastRenewed, 3));
      }
    } else {
      nextRenewal = toDateOnlyUTC(addYearsUTC(lastRenewed, 3));
    }

    const suspended = nextRenewal.getTime() <= now.getTime();

    memorial.lastRenewed = lastRenewed;
    memorial.nextRenewal = nextRenewal;
    memorial.suspended = suspended;

    await memorial.save();

    return res.json({
      message: "Memorial renewal updated",
      memorial,
    });
  } catch (err) {
    console.error("updateMemorialRenewal error", err);
    return res.status(500).json({ message: "Internal Server Error" });
  }
};
module.exports = {
  index,
  getAllMemorialAdmin,
  getAllMemorial,
  getPinnedMemorials,
  updateProfileImageMemorial,
  updateCover,
  createMemorial,
  deleteMemorial,
  getMemorialsByUserId,
  getMemorialById,
  getMemorialsByAddress,
  updateMemorial,
  suspendMemorial,
  unsuspendMemorial,
  pinMemorial,
  unpinMemorial,
  addView,
  getByCategory,
  updateMemorialRenewal,
  updateCoverPhotoMemorial
};
