const { Op } = require("sequelize");
const Obituary = require("../models/Obituary");
const User = require("../models/User");
const ObituaryPurchase = require("../models/obituaryPurchase");
const cron = require("node-cron");
const Notification = require("../models/Notification");
const ObituariesAudioBG = require("../models/ObituariesAudioBG");
const fs = require("fs");
const path = require("path");
const index = async (req, res) => {
  try {
    const obituaries = await Obituary.findAll();
    console.log({ obituaries });
    return res.json(obituaries);
  } catch (error) {
    console.log(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const getAllObituary = async (req, res) => {
  try {
    const page = req.query.page ? req.query.page : 1;
    const pageSize = req.query.pageSize ? req.query.pageSize : 4;
    const type = req.query.type ? req.query.type : null;

    const obituaries = await Obituary.findAndCountAll({
      where: {
        ...(req.query.type && {
          pinned: type === "pinned" ? 1 : 0,
        }),
        ...(req.query.search && {
          [Op.or]: [
            { Firstname: { [Op.regexp]: req.query.search } },
            { MiddleName: { [Op.regexp]: req.query.search } },
            { LastName: { [Op.regexp]: req.query.search } },
          ],
        }),
      },
      include: [
        {
          model: User,
          as: "user",
          attributes: {
            exclude: ["password"],
          },
        },
      ],
      order: [["createdOn", "DESC"]],
      limit: parseInt(pageSize),
      offset: (parseInt(page) - 1) * parseInt(pageSize),
    });
    const total = Math.ceil(obituaries.count / pageSize);
    console.log({ obituaries });
    return res.status(201).json({
      filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
      obituaries: obituaries.rows,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};
const getAdminObituaries = async (req, res) => {
  try {
    const page = req.query.page ? parseInt(req.query.page) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize) : 4;
    const type = req.query.type || null;

    // ✅ Step 1: get all admin user IDs
    const admins = await User.findAll({
      where: { type: "admin" },
      attributes: ["userid"], // adjust to your real PK (e.g. "id" or "userid")
    });

    const adminIds = admins.map((u) => u.userid);

    if (adminIds.length === 0) {
      return res.status(200).json({
        filters: { page, nextPage: page + 1, total: 0 },
        obituaries: [],
      });
    }

    // ✅ Step 2: build where condition for obituaries
    let whereCondition = {
      user_id: { [Op.in]: adminIds }, // only admin obituaries
    };

    if (req.query.type) {
      whereCondition = {
        ...whereCondition,
        pinned: type === "pinned" ? 1 : 0,
      };
    }

    if (req.query.search) {
      whereCondition = {
        ...whereCondition,
        [Op.or]: [
          { Firstname: { [Op.regexp]: req.query.search } },
          { MiddleName: { [Op.regexp]: req.query.search } },
          { LastName: { [Op.regexp]: req.query.search } },
        ],
      };
    }

    // ✅ Step 3: query obituaries
    const obituaries = await Obituary.findAndCountAll({
      where: whereCondition,
      include: [
        {
          model: User,
          as: "user",
          attributes: { exclude: ["password"] },
        },
      ],
      order: [["createdOn", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
    });

    const total = Math.ceil(obituaries.count / pageSize);

    return res.status(200).json({
      filters: { page, nextPage: page + 1, total },
      obituaries: obituaries.rows,
    });
  } catch (error) {
    console.error("❌ getAdminObituaries error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};



const getPinnedObituaries = async (req, res) => {
  try {
    const obituaries = await Obituary.findAll({
      where: { pinned: true, suspended: false },
      order: [["createdOn", "DESC"]],
    });
    return res.json(obituaries);
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const pinObituary = async (req, res) => {
  try {
    const obituary = await Obituary.findByPk(req.params.id);
    obituary.pinned = true;
    await obituary.save();
    return res.json({
      message: "Pinned the obituary",
      pinned: obituary.pinned,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};
const unpinObituary = async (req, res) => {
  try {
    const obituary = await Obituary.findByPk(req.params.id);
    obituary.pinned = false;
    await obituary.save();
    return res.json({
      message: "Unpinned the obituary",
      pinned: obituary.pinned,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const upDateProfileImageObituaries = async (req, res) => {
  try {
    const { memorialId } = req.body;
    console.log({ memorialId });
    const imagePath = req.file.path;
    const memorial = await Obituary.update(
      { ProfileImgUrl: imagePath },
      {
        where: {
          id: memorialId,
        },
      }
    );

    res.status(200).json({ message: "Upload successful", memorial });
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const createObituary = async (req, res) => {
  try {
    let fileData = {};

    if (req.files) {
      // ✅ Profile image
      if (req.files.ProfileImgUrl?.[0]) {
        fileData.ProfileImgUrl = req.files.ProfileImgUrl[0].path
          .replace(/\\/g, "/")
          .replace("public/", "");
      }

      // ✅ Slide image
      if (req.files.slideImage?.[0]) {
        fileData.SlideImage = req.files.slideImage[0].path
          .replace(/\\/g, "/")
          .replace("public/", "");
      }

      // ✅ Gallery images
      if (req.files.images?.length > 0) {
        req.files.images.forEach((file, idx) => {
          fileData[`image${idx + 1}`] = file.path
            .replace(/\\/g, "/")
            .replace("public/", "");
        });
      }
    }

    // ✅ Create obituary
    const obituary = await Obituary.create({
      ...req.body,
      user_id: req.body.userId,
      CreateBy: req.body.userId,
      obituaryTitle:
        req.body.obituaryTitle ||
        `${req.body.Firstname || ""} ${req.body.LastName || ""}`.trim() ||
        "Untitled Obituary",
      pinned: req.body.type === "pinned",
      daily: req.body.type === "daily",
      ...fileData,
    });

    // ✅ Save audio if uploaded
    if (req.files?.audio?.[0]) {
      const audioPath = req.files.audio[0].path
        .replace(/\\/g, "/")
        .replace("public/", "");

      await ObituariesAudioBG.create({
        audio: audioPath,
        obituaries_id: obituary.id,
      });
    }

    // ✅ Notification
    await Notification.create({
      text: "An obituary is being created",
      user_id: req.body.userId,
      type: "obituary",
    });

    return res.status(201).json({
      message: "Obituary created successfully",
      obituary,
    });
  } catch (error) {
    console.error("❌ createObituary error:", error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const uploadProfileImageObetrius = async (req, res) => {
  try {
    const { obituaryId } = req.body;

    const files = req.files;

    console.log({ uploadedFiles: files });

    let updateData = {};

    if (files?.file?.[0]) {
      let profilePath = files.file[0].path.replace("public", "api");
      updateData.ProfileImgUrl = profilePath;
    }

    if (files?.slideImage?.[0]) {
      console.log("Slide Image File:", files.slideImage[0]);
      let slidePath = files.slideImage[0].path.replace("public", "api");
      updateData.SlideImage = slidePath;
    }

    const updated = await Obituary.update(updateData, {
      where: {
        id: obituaryId,
      },
    });

    return res
      .status(200)
      .json({ message: "Images uploaded successfully", updated });
  } catch (error) {
    console.error("Upload error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const getObituaryByUserId = async (req, res) => {
  const { userId } = req.params;

  try {
    const page = req.query.page ? parseInt(req.query.page) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize) : 4;

    // 🔎 Fetch the user who is making the request
    const user = await User.findByPk(userId);

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    let whereCondition = {};

    if (user.type !== "obituary_admin") {
      // Normal user → restrict to their own obituaries
      whereCondition = { user_id: userId };
    }
    // If obituary_admin → leave whereCondition empty (fetch ALL)

    const obituaries = await Obituary.findAndCountAll({
      where: whereCondition,
      order: [["createdOn", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
    });

    const total = Math.ceil(obituaries.count / pageSize);

    return res.status(200).json({
      filters: { page, nextPage: page + 1, total },
      obituaries: obituaries.rows,
    });
  } catch (error) {
    console.error("❌ getObituaryByUserId error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

const deleteObituary = async (req, res) => {
  try {
    const { id } = req.params;
    const obituary = await Obituary.findByPk(id);

    if (!obituary) {
      return res.status(404).json({ error: "Obituary not found" });
    }
    await ObituaryPurchase.destroy({ where: { obituary_id: id } });

    await obituary.destroy();
    return res.status(200).json({ message: "Obituary deleted successfully" });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const getObituaryById = async (req, res) => {
  try {
    const { id } = req.params;

    const obituary = await Obituary.findByPk(id, {
      include: [
        {
          model: User,
          as: "user",
        },
        {
          model: ObituariesAudioBG,
          as: "audioBg", // alias must match the relation
          attributes: ["id", "audio"],
        },
      ],
    });

    if (!obituary) {
      return res.status(404).json({ error: "Obituary not found" });
    }

    res.status(200).json(obituary);
  } catch (error) {
    console.error("❌ getObituaryById error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};


const getByAddress = async (req, res) => {
  const { address } = req.params;
  try {
    // Find the memorial by its ID
    const memorial = await Obituary.findOne({
      where: { MemorialAddress: address },
    });
    if (!memorial) {
      return res.status(404).json({ error: "Memorial not found" });
    }
    res.status(200).json(memorial);
  } catch (error) {
    console.error(error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};




const updateObituary = async (req, res) => {
  try {
    const { id } = req.params;

    let updateData = { ...req.body };

    // ✅ Handle image uploads
    if (req.files) {
      // Profile image
      if (req.files.ProfileImgUrl && req.files.ProfileImgUrl[0]) {
        updateData.ProfileImgUrl = req.files.ProfileImgUrl[0].path
          .replace(/\\/g, "/")
          .replace("public/", "");
      }

      // Slide image
      if (req.files.slideImage && req.files.slideImage[0]) {
        updateData.SlideImage = req.files.slideImage[0].path
          .replace(/\\/g, "/")
          .replace("public/", "");
      }

      // Gallery images
      if (req.files.images && req.files.images.length > 0) {
        req.files.images.forEach((file, idx) => {
          updateData[`image${idx + 1}`] = file.path
            .replace(/\\/g, "/")
            .replace("public/", "");
        });
      }
    }

    // ✅ Update obituary
    const [updated] = await Obituary.update(updateData, { where: { id } });

    // ✅ Handle audio file
    if (req.files && req.files.audio && req.files.audio[0]) {
      const newAudioPath = req.files.audio[0].path
        .replace(/\\/g, "/")
        .replace("public/", "");

      // Find existing audio record
      const existingAudio = await ObituariesAudioBG.findOne({
        where: { obituaries_id: id },
      });

      if (existingAudio) {
        // Optional: delete old file from server
        const oldFilePath = path.join("public", existingAudio.audio);
        if (fs.existsSync(oldFilePath)) {
          fs.unlinkSync(oldFilePath);
        }

        // Delete old record
        await ObituariesAudioBG.destroy({ where: { obituaries_id: id } });
      }

      // Create new audio record
      await ObituariesAudioBG.create({
        obituaries_id: id,
        audio: newAudioPath,
      });
    }

    // ✅ Get obituary with relations
    const obituary = await Obituary.findByPk(id, {
      include: [{ model: ObituariesAudioBG, as: "audioBg", attributes: ["id", "audio"] }],
    });

    return res.status(200).json({
      message: "Obituary updated successfully",
      obituary,
    });
  } catch (error) {
    console.error("❌ updateObituary error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

// Function to update status of a specific record by ID
const updatePinById = async (id, days) => {
  const daysAgo = new Date(Date.now() - days * 24 * 60 * 60 * 1000);

  const [updatedCount] = await Obituary.update(
    { pinned: false },
    {
      where: {
        id: id,
        pinned: true,
        CreatedOn: {
          [Op.lt]: daysAgo, // Only update if created more than 5 days ago
        },
      },
    }
  );

  if (updatedCount > 0) {
    console.log(`Updated status to false for record with ID: ${id}`);
  }
};

const suspendObituary = async (req, res) => {
  try {
    const obituary = await Obituary.findOne({ where: { id: req.params.id } });
    obituary.suspended = true;
    await obituary.save();
    return res.json({
      message: "Suspended the obituary successfully!",
      suspended: obituary.suspended,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const unsuspendObituary = async (req, res) => {
  try {
    const obituary = await Obituary.findOne({ where: { id: req.params.id } });
    obituary.suspended = false;
    await obituary.save();
    return res.json({
      message: "Unsuspended the obituary successfully!",
      suspended: obituary.suspended,
    });
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

const addView = async (req, res) => {
  try {
    const obituary = await Obituary.findOne({ where: { id: req.params.id } });
    obituary.views++;
    await obituary.save();
    return res.json(obituary.views);
  } catch (error) {
    console.error(error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
};

// Scheduled job to update status for all records
const schedulePinUpdate = () => {
  cron.schedule("0 0 * * *", async () => {
    // Runs every day at midnight
    const records = await Obituary.findAll({
      where: {
        pinned: true,
        // CreatedOn: {
        //   [Op.lt]: new Date(Date.now() - days * 24 * 60 * 60 * 1000), // Find records older than days days
        //   // [Op.lt]: new Date(Date.now() - days * 1000), // Find records older than 5 days
        // },
      },
    });

    for (const record of records) {
      await updatePinById(record.id, record.days); // Update each record's status
    }
  });
};

// Function to update status of a specific record by ID
const updateDailyById = async (id, days) => {
  const daysAgo = new Date(Date.now() - days * 24 * 60 * 60 * 1000);

  const [updatedCount] = await Obituary.update(
    { daily: false },
    {
      where: {
        id: id,
        daily: true,
        CreatedOn: {
          [Op.lt]: daysAgo, // Only update if created more than 5 days ago
        },
      },
    }
  );

  if (updatedCount > 0) {
    console.log(`Updated status to false for record with ID: ${id}`);
  }
};

// Scheduled job to update status for all records
const scheduleDailyUpdate = () => {
  cron.schedule("0 0 * * *", async () => {
    // Runs every day at midnight
    const records = await Obituary.findAll({
      where: {
        daily: true,
        // CreatedOn: {
        //   // [Op.lt]: new Date(Date.now() - 5 * 24 * 60 * 60 * 1000), // Find records older than 5 days
        //   [Op.lt]: new Date(Date.now() - 5 * 1000), // Find records older than 5 days
        // },
      },
    });

    for (const record of records) {
      await updateDailyById(record.id, record.days); // Update each record's status
    }
  });
};
// Get counts of pinned and daily obituaries
const getObituaryPlanCounts = async (req, res) => {
  try {
    const pinnedCount = await Obituary.count({ where: { type: "pinned" } });
    const dailyCount = await Obituary.count({ where: { type: "daily" } });

    return res.json({ pinnedCount, dailyCount });
  } catch (error) {
    console.error("Plan counts error:", error);
    return res
      .status(500)
      .json({ error: "Error fetching obituary plan counts" });
  }
};

// Get obituaries by plan (pinned or daily)
const getObituariesByPlan = async (req, res) => {
  try {
    const { plan } = req.params; // "pinned" or "daily"

    if (!["pinned", "daily"].includes(plan)) {
      return res.status(400).json({ error: "Invalid plan type" });
    }

    const obituaries = await Obituary.findAll({
      where: { type: plan },
      include: [{ model: User, as: "user" }],
      order: [["CreatedOn", "DESC"]],
    });

    return res.json(obituaries);
  } catch (error) {
    console.error("Obituaries by plan error:", error);
    return res.status(500).json({ error: "Error fetching obituaries by plan" });
  }
};

module.exports = {
  index,
  createObituary,
  getAllObituary,
  getPinnedObituaries,
  deleteObituary,
  getObituaryByUserId,
  getAdminObituaries,
  uploadProfileImageObetrius,
  getObituaryById,
  updateObituary,
  upDateProfileImageObituaries,
  getByAddress,
  pinObituary,
  unpinObituary,
  schedulePinUpdate,
  scheduleDailyUpdate,
  suspendObituary,
  unsuspendObituary,
  addView,
  getObituaryPlanCounts,
  getObituariesByPlan,
};
