// controller/ServiceDetailController.js
const ServiceDetail = require("../models/ServiceDetail"); // model ka naam tumne file me Journey.js rakha hai
const User = require("../models/User");
const Memorial = require("../models/Memorial");

const isInvalidDate = (v) =>
  !v || v === "0000-00-00" || Number.isNaN(new Date(v).getTime());

const toDate = (v) => new Date(v);

const toDateOnlyUTC = (date) =>
  new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate()));

const addYearsUTC = (date, years) => {
  const d = new Date(date);
  d.setUTCFullYear(d.getUTCFullYear() + years);
  return d;
};

module.exports = {
  async create(req, res) {
    try {
      const body = { ...req.body };

      // Helpers de data
      const isInvalidDate = (v) =>
        !v || v === "0000-00-00" || Number.isNaN(new Date(v).getTime());

      const toDateOnlyUTC = (date) =>
        new Date(Date.UTC(date.getUTCFullYear(), date.getUTCMonth(), date.getUTCDate()));

      const addYearsUTC = (date, years) => {
        const d = new Date(date);
        d.setUTCFullYear(d.getUTCFullYear() + years);
        return d;
      };

      // Usuário que está postando o serviço
      const loggedInUserId = body.user_id;
      const serviceProviderId = body.service_id;

      const user = await User.findByPk(loggedInUserId);
      if (!user) {
        return res.status(404).json({ message: "User not found" });
      }

      const provider = await User.findByPk(serviceProviderId);
      if (!provider) {
        return res.status(404).json({ message: "Service provider not found" });
      }

      // Upload de imagem se existir
      if (req.file) {
        body.image = req.file.path;
      }

      // Datas de renovação para ServiceDetail com ciclo de 1 ano
      const now = new Date();

      let lastRenewed = !isInvalidDate(body.lastRenewed)
        ? toDateOnlyUTC(new Date(body.lastRenewed))
        : toDateOnlyUTC(now);

      let nextRenewal;
      if (!isInvalidDate(body.nextRenewal)) {
        const provided = toDateOnlyUTC(new Date(body.nextRenewal));
        nextRenewal =
          provided.getTime() > lastRenewed.getTime()
            ? provided
            : toDateOnlyUTC(addYearsUTC(lastRenewed, 1));
      } else {
        nextRenewal = toDateOnlyUTC(addYearsUTC(lastRenewed, 1));
      }

      // Criação do registro
      const serviceDetail = await ServiceDetail.create({
        user_id: loggedInUserId,
        service_id: serviceProviderId,
        title: body.title,
        type: body.type,
        description: body.description,
        image: body.image || null,
        lastRenewed,         // novo campo
        nextRenewal,         // novo campo
      });

      return res.json(serviceDetail);
    } catch (err) {
      console.error("Error creating service detail:", err);
      return res.status(500).json({ message: "Server error" });
    }
  }
  ,

  async getAll(req, res) {
    try {
      const serviceDetails = await ServiceDetail.findAll({
        include: [
          { model: User, as: "user", attributes: ["userid", "firstName", "lastName", "email", "service_id"] },
          { model: User, as: "serviceId", attributes: ["userid", "firstName", "lastName", "email", "service_id"] }
        ],
        order: [["createdAt", "DESC"]], // newest first
      });

      return res.json(serviceDetails);
    } catch (err) {
      console.error("Error fetching service details:", err);
      return res.status(500).json({ message: "Server error" });
    }
  },

  // GET single service detail by ID
  async getById(req, res) {
    try {
      const { id } = req.params;

      const serviceDetail = await ServiceDetail.findAll({
        where: { service_id: id }, // yaha id service provider ka ID hoga
        include: [
          { model: User, as: "user", attributes: ["userid", "firstName", "lastName", "email", "service_id"] },
          { model: User, as: "serviceId", attributes: ["userid", "firstName", "lastName", "email", "service_id"] }
        ],
      });


      if (!serviceDetail) {
        return res.status(404).json({ message: "Service detail not found" });
      }

      return res.json(serviceDetail);
    } catch (err) {
      console.error("Error fetching service detail:", err);
      return res.status(500).json({ message: "Server error" });
    }
  }
  ,

  async updateserviceRenewal(req, res) {
    try {
      const id = req.params.id;
      const body = req.body || {};
      const now = toDateOnlyUTC(new Date());

      const memorial = await ServiceDetail.findByPk(id);
      if (!memorial) {
        return res.status(404).json({ message: "Memorial not found" });
      }


      let lastRenewed;
      if (!isInvalidDate(body.lastRenewed)) {
        lastRenewed = toDateOnlyUTC(toDate(body.lastRenewed));
      } else if (!isInvalidDate(memorial.lastRenewed)) {
        lastRenewed = toDateOnlyUTC(toDate(memorial.lastRenewed));
      } else if (!isInvalidDate(memorial.CreatedOn)) {
        lastRenewed = toDateOnlyUTC(toDate(memorial.CreatedOn));
      } else {
        lastRenewed = now;
      }

      // Resolve nextRenewal
      let nextRenewal = null;
      if (!isInvalidDate(body.nextRenewal)) {
        nextRenewal = toDateOnlyUTC(toDate(body.nextRenewal));
        if (nextRenewal.getTime() <= lastRenewed.getTime()) {
          nextRenewal = toDateOnlyUTC(addYearsUTC(lastRenewed, 3));
        }
      } else {
        nextRenewal = toDateOnlyUTC(addYearsUTC(lastRenewed, 3));
      }

      const suspended = nextRenewal.getTime() <= now.getTime();

      memorial.lastRenewed = lastRenewed;
      memorial.nextRenewal = nextRenewal;
      memorial.suspended = suspended;

      await memorial.save();

      return res.json({
        message: "Memorial renewal updated",
        memorial,
      });
    } catch (err) {
      console.error("updateMemorialRenewal error", err);
      return res.status(500).json({ message: "Internal Server Error" });
    }
  }
  ,

  async deleteService(req, res) {
    try {
      // Get the service provider's ID from the request params
      const { id } = req.params;

      // Check if the service provider exists in the database
      const serviceDetail = await ServiceDetail.findByPk(id);
      if (!serviceDetail) {
        return res.status(404).json({ message: "Service provider not found" });
      }

      // Deleting the service detail
      await serviceDetail.destroy();

      // Respond with a success message
      return res.status(200).json({ message: "Service provider deleted successfully" });
    } catch (err) {
      console.error("Error deleting service provider:", err);
      return res.status(500).json({ message: "Server error" });
    }
  }

};


