const jwt = require("jsonwebtoken");
const bcrypt = require("bcrypt");
const { Op } = require("sequelize");
const nodemailer = require("nodemailer");

const { sendMail } = require("./mailController");
const User = require("../models/User");
const Notification = require("../models/Notification");

module.exports = {
  // List all users (no pagination)
  async index(req, res) {
    try {
      const users = await User.findAll();
      return res.json(users);
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Sign up new user
  async signUp(req, res) {
    try {
      const { email, password, type, service_id } = req.body;

      const foundUser = await User.findOne({ where: { email } });
      if (foundUser) {
        return res.status(400).json({ error: "Email already registered" });
      }

      const salt = bcrypt.genSaltSync(10);
      const hashedPassword = bcrypt.hashSync(password, salt);

      const user = await User.create({
        ...req.body,
        password: hashedPassword,
        service_id: type === "service_provider" ? service_id : null,
      });

      // Optional notification
      if (Notification) {
        await Notification.create({
          text: "A user is being created",
          user_id: user.userid,
          type: "user",
        });
      }

      return res.status(201).json({
        message: "User registered successfully",
        user,
      });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Login
  async signIn(req, res) {
    try {
      const { email, password } = req.body;
      const user = await User.findOne({ where: { email } });
      if (!user) {
        return res.status(400).json({ message: "Invalid Email" });
      }

      const isMatch = await bcrypt.compare(password, user.password);
      if (!isMatch) {
        return res.status(400).json({ message: "Invalid credentials" });
      }

      const token = jwt.sign(
        {
          userid: user.userid,
          email: user.email,
          type: user.type,
          firstName: user.firstName,
          lastName: user.lastName,
        },
        process.env.SECRETKEY,
        { expiresIn: "1d" }
      );

      return res.status(200).json({ token, user });
    } catch (error) {
      console.error(error);
      res.status(500).json({ message: "Internal Server Error" });
    }
  },

  // Get one user
  async getSingleUser(req, res) {
    try {
      const user = await User.findOne({ where: { userid: req.params.id } });
      if (!user) {
        return res
          .status(400)
          .json({ message: "User with that ID does not exist!" });
      }
      return res.json(user);
    } catch (error) {
      console.error(error);
      return res
        .status(500)
        .json({ result: false, error: "Internal Server Error" });
    }
  },

  // Update user
  async updateUser(req, res) {
    try {
      const { id } = req.params;
      const {
        firstName,
        email,
        lastName,
        type,
        companydescription,
        companyname,
        newPassword,
        confirmPassword,
        town,
        state,
        country,
      } = req.body;

      const user = await User.findByPk(id);
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }

      user.firstName = firstName || user.firstName;
      user.lastName = lastName || user.lastName;
      user.email = email || user.email;
      user.type = type || user.type;
      user.town = town;
      user.state = state;
      user.country = country;
      if (user.type === "service_provider") {
        user.companyname = companyname;
        user.companydescription = companydescription;
      }
      if (newPassword && confirmPassword) {
        if (newPassword !== confirmPassword) {
          return res.status(400).json({ message: "Passwords do not match!" });
        }
        const salt = bcrypt.genSaltSync(10);
        const hashedPassword = bcrypt.hashSync(newPassword, salt);
        user.password = hashedPassword;
      }

      await user.save();
      res.status(200).json({ message: "User updated successfully", user });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Update type only
  async updateUserType(req, res) {
    try {
      const user = await User.findByPk(req.params.id);
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }
      user.type = req.body.type;
      await user.save();
      return res.json({ message: "User type updated!" });
    } catch (err) {
      console.error(err);
      return res.status(500).json({ message: "Internal Server Error" });
    }
  },

  // Forgot password (send email)
  async forgotPassword(req, res) {
    try {
      const { email } = req.body;

      const user = await User.findOne({ where: { email } });
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }

      const resetToken = jwt.sign(
        { userId: user.userid },
        process.env.RESET_SECRET || "reset_secret_key",
        { expiresIn: "1h" }
      );

      const resetLink = `http://your-frontend-app/reset-password/${resetToken}`;
      const transporter = nodemailer.createTransport({
        service: "gmail",
        auth: {
          user: process.env.EMAIL_USER,
          pass: process.env.EMAIL_PASS,
        },
      });

      await transporter.sendMail({
        from: process.env.EMAIL_USER,
        to: user.email,
        subject: "Password Reset",
        text: `Click on the following link to reset your password: ${resetLink}`,
      });

      res
        .status(200)
        .json({ message: "Password reset email sent successfully" });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Reset password
  async resetPassword(req, res) {
    try {
      const { token } = req.params;
      const { newPassword } = req.body;

      const decoded = jwt.verify(
        token,
        process.env.RESET_SECRET || "reset_secret_key"
      );
      const user = await User.findByPk(decoded.userId);
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }

      const salt = bcrypt.genSaltSync(10);
      user.password = bcrypt.hashSync(newPassword, salt);
      await user.save();

      res.status(200).json({ message: "Password reset successfully" });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Get all users (with pagination + search)
  async allUsers(req, res) {
    try {
      const page = parseInt(req.query.page) || 1;
      const pageSize = parseInt(req.query.pageSize) || 4;

      const users = await User.findAndCountAll({
        where: {
          ...(req.query.search && {
            [Op.or]: [
              { email: { [Op.like]: `%${req.query.search}%` } },
              { firstName: { [Op.like]: `%${req.query.search}%` } },
              { lastName: { [Op.like]: `%${req.query.search}%` } },
            ],
          }),
        },
        limit: pageSize,
        offset: (page - 1) * pageSize,
      });

      const total = Math.ceil(users.count / pageSize);

      return res.status(200).json({
        filters: { page, nextPage: page + 1, total },
        users: users.rows,
      });
    } catch (error) {
      console.error(error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },
  async allProviders(req, res) {
    try {
      const page = parseInt(req.query.page) || 1;
      const pageSize = parseInt(req.query.pageSize) || 4;
      const { search, service_id } = req.query;

      const where = {
        type: "service_provider", // only service providers
      };

      // Optional search
      if (search) {
        where[Op.or] = [
          { email: { [Op.like]: `%${search}%` } },
          { firstName: { [Op.like]: `%${search}%` } },
          { lastName: { [Op.like]: `%${search}%` } },
        ];
      }

      // Optional service_id filter
      if (service_id) {
        where.service_id = service_id;
      }

      const users = await User.findAndCountAll({
        where,
        limit: pageSize,
        offset: (page - 1) * pageSize,
      });

      const total = Math.ceil(users.count / pageSize);

      return res.status(200).json({
        filters: { page, nextPage: page + 1, total },
        users: users.rows,
      });
    } catch (error) {
      console.error("❌ allProviders error:", error);
      return res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Delete user
  async deleteUserById(req, res) {
    try {
      const user = await User.findByPk(req.params.id);
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }
      await user.destroy();
      res.status(200).json({ message: "User deleted successfully" });
    } catch (error) {
      console.error(error);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Search users by email
  async findUserByEmail(req, res) {
    try {
      const users = await User.findAll({
        where: { email: { [Op.like]: `%${req.query.email}%` } },
      });
      return res.json(users);
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },

  // Get allowed types (from ENUM)
  async getUserTypes(req, res) {
    try {
      let values = User.rawAttributes.type.values;

      // remove "super_admin"
      values = values.filter((val) => val !== "super_admin");

      return res.json(values);
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: "Internal Server Error" });
    }
  },
  async getAdminsAndServiceProviders(req, res) {
    try {
      const users = await User.findAll({
        where: { type: ["admin", "service_provider"] },
        attributes: ["userid", "firstName", "lastName", "email", "type"],
      });

      return res.json(users);
    } catch (err) {
      console.error("Error fetching admins and service providers:", err);
      return res.status(500).json({ error: "Failed to fetch users" });
    }
  },
};
