import ServiceProvider from '../models/ServiceProvider.js';
import Service from '../models/Service.js';
import ProviderService from '../models/ProviderService.js';

export const updateServiceProvider = async (req, res) => {
  try {
    const { id } = req.params;
    const updated = await ServiceProvider.findByIdAndUpdate(id, { $set: req.body }, { new: true });

    if (!updated) return res.status(404).json({ success: false, message: "Provider not found" });

    res.status(200).json({ success: true, data: updated });
  } catch (err) {
    console.error("Update Error:", err);
    res.status(500).json({ success: false, message: "Failed to update provider" });
  }
};


export const saveServiceProvider = async (req, res) => {
  try {
    const {
      name,
      description,
      phone,
      email,
      clientsAtOnce,
      showOnBooking,
    } = req.body;

    if (!name || !description) {
      return res.status(400).json({ message: 'Name and description are required' });
    }

    const provider = new ServiceProvider({
      name,
      description,
      phone,
      email,
      clientsAtOnce,
      showOnBooking,
    });

    await provider.save();

    res.status(200).json({ message: 'Provider saved successfully', provider });
  } catch (error) {
    console.error('Error saving provider:', error);
    res.status(500).json({ error: 'Server error' });
  }
};



export const getService = async (req, res) => {
  try {
    const services = await Service.find();
    res.status(200).json(services);
  } catch (error) {
    console.error('Error fetching services:', error);
    res.status(500).json({ message: 'Failed to fetch services' });
  }
};



export const updateProviderServices = async (req, res) => {
  try {
    const { id } = req.params;
    const updated = await ProviderService.findByIdAndUpdate(id, {
      $set: req.body,
    }, { new: true });

    if (!updated) return res.status(404).json({ message: "Not found" });

    res.status(200).json({ success: true, data: updated });
  } catch (err) {
    console.error("Update error:", err);
    res.status(500).json({ success: false, message: "Update failed" });
  }
};


export const saveProviderServices = async (req, res) => {
  const { providerId, services } = req.body;

  if (!providerId || !Array.isArray(services)) {
    return res.status(400).json({ message: 'providerId and services are required.' });
  }

  try {
    // Check if a record already exists for this provider
    const existing = await ProviderService.findOne({ providerId });

    if (existing) {
      // Update existing record
      existing.services = services;
      await existing.save();
      return res.status(200).json({ message: 'Services updated successfully.', data: existing });
    }

    // Create new record
    const newRecord = await ProviderService.create({ providerId, services });
    return res.status(201).json({ message: 'Services saved successfully.', data: newRecord });

  } catch (error) {
    console.error('Error saving provider services:', error);
    res.status(500).json({ message: 'Internal server error.' });
  }
};



export const deleteService = async (req, res) => {
  const serviceId = req.params.id;

  try {
    const deleted = await ServiceProvider.findByIdAndDelete(serviceId);

    if (!deleted) {
      return res.status(404).json({ success: false, message: 'Service not found' });
    }

    return res.status(200).json({ success: true, message: 'Service deleted successfully' });
  } catch (error) {
    console.error('Error deleting service:', error);
    return res.status(500).json({ success: false, message: 'Server error' });
  }
};