import Service from '../models/Service.js';
import ServiceTime from '../models/ServiceTime.js';
import ServicePageProvider from '../models/ServicePageProvider.js';
import Category from '../models/Category.js';
import CategoryService from '../models/CategoryService.js'
import mongoose from 'mongoose';


export const getServiceById = async (req, res) => {
  try {
    const service = await Service.findById(req.params.id);
    if (!service) {
      return res.status(404).json({ message: 'Service not found' });
    }

    const serviceTime = await ServiceTime.findOne({ categoryId: req.params.id });

    const servicePageProviders = await ServicePageProvider.find({ ServicePageId: req.params.id })
      .populate('ServicePageId'); // <-- populate Service data if needed

    res.status(200).json({
      service,
      serviceTime,
      servicePageProviders, // ✅ also returning providers
    });
  } catch (error) {
    console.error('Error fetching service by ID:', error);
    res.status(500).json({ message: 'Failed to fetch service' });
  }
};



export const updateService = async (req, res) => {
  try {
    const { id } = req.params;
    const { name, description, display, price, tax, image } = req.body;

    const updatedService = await Service.findByIdAndUpdate(
      id,
      { name, description, display, price, tax, image },
      { new: true, runValidators: true }
    );

    if (!updatedService) {
      return res.status(404).json({ message: 'Service not found' });
    }

    res.status(200).json(updatedService);
  } catch (error) {
    console.error('Error updating service:', error);
    res.status(500).json({ message: 'Failed to update service' });
  }
};


export const GetsaveCategoryServices = async (req, res) => {
  try {
    const categories = await Category.find();

    const categoriesWithServices = await Promise.all(
      categories.map(async (category) => {
        const categoryServiceDocs = await CategoryService.find({ categoryId: category._id }).populate('services');

        // flatten all service arrays from multiple documents into a single array
        const allServices = categoryServiceDocs.flatMap(serviceDoc =>
          serviceDoc.services.map(service => 
          ({
            _id: service._id,
            name: service.name,
            price:service.price,
            
          }))
        );

        return {
          ...category._doc,
          services: allServices
        };
      })
    );

    res.json(categoriesWithServices);
  } catch (err) {
    console.error("Error fetching categories with services", err);
    res.status(500).json({ error: "Failed to fetch categories with services" });
  }
};



export const createService = async (req, res) => {
  try {
    const { name, description, display, price, tax, image } = req.body;

    const newCategory = new Service({
      name,
      description,
      display,
      price,
      tax,
      image,
    });

    const savedCategory = await newCategory.save();
    res.status(201).json(savedCategory);
  } catch (error) {
    console.error('Error creating category:', error);
    res.status(500).json({ message: 'Failed to create category' });
  }
};


export const createServiceTime = async (req, res) => {
  try {
    const serviceTime = new ServiceTime(req.body);
    const saved = await serviceTime.save();
    res.status(201).json({ status: true, message: "Saved successfully", data: saved });
  } catch (error) {
    console.error("Error saving service time:", error);
    res.status(500).json({ status: false, message: "Failed to save", error });
  }
};




export const ServicePage = async (req, res) => {
  console.log("running");
  
  try {
    const { ServicePageId, providers } = req.body;
console.log("running",providers);
    // Check if already exists (update if needed)
    let record = await ServicePageProvider.findOne({ ServicePageId });

    if (record) {
      record.providers = providers;
      await record.save();
    } else {
      record = new ServicePageProvider({ ServicePageId, providers });
      await record.save();
    }

    res.status(200).json({ message: 'Location providers saved successfully', data: record });
  } catch (error) {
    console.error('Error saving providers:', error);
    res.status(500).json({ error: 'Server error' });
  }
};



export const Categoryregistration =async (req, res) => {
   try {
    const { name, description, position, isVisible, isDefault, image } = req.body;

    const newCategory = new Category({
      name,
      description,
      position,
      isVisible: isVisible === 'true',
      isDefault: isDefault === 'true',
      image: image || 'No image uploaded',
    });

    await newCategory.save();

    res.status(201).json({ message: 'Category created successfully', category: newCategory });
  } catch (error) {
    console.error('Category creation error:', error);
    res.status(500).json({ message: 'Server error creating category' });
  }
};


export const updateCategory = async (req, res) => {
  try {
    const categoryId = req.params.id;

    const { name, description, position, isVisible, isDefault, image } = req.body;

    const updatedCategory = await Category.findByIdAndUpdate(
      categoryId,
      {
        name,
        description,
        position,
        isVisible: isVisible === 'true' || isVisible === true,
        isDefault: isDefault === 'true' || isDefault === true,
        image: image || 'No image uploaded',
      },
      { new: true }
    );

    if (!updatedCategory) {
      return res.status(404).json({ message: 'Category not found' });
    }

    res.status(200).json({ message: 'Category updated successfully', category: updatedCategory });
  } catch (error) {
    console.error('Category update error:', error);
    res.status(500).json({ message: 'Server error updating category' });
  }
};



export const updateCategoryServices = async (req, res) => {
  try {
    const categoryId = req.params.id;
    const { services } = req.body;

    if (!categoryId || !services || !Array.isArray(services)) {
      return res.status(400).json({ message: "Category ID and services array are required." });
    }

    const serviceIds = services.map(s => {
      const id = typeof s === 'string' ? s : s._id;
      if (!mongoose.Types.ObjectId.isValid(id)) {
        throw new Error(`Invalid service ID: ${id}`);
      }
      return new mongoose.Types.ObjectId(id);
    });

    const updated = await CategoryService.findOneAndUpdate(
      { categoryId: new mongoose.Types.ObjectId(categoryId) },
      { services: serviceIds },
      { new: true }
    );

    if (!updated) {
      const newService = new CategoryService({
        categoryId: new mongoose.Types.ObjectId(categoryId),
        services: serviceIds
      });

      await newService.save();
      return res.status(201).json({ message: 'Service added successfully', service: newService });
    }

    res.status(200).json({ message: "Category service updated successfully.", service: updated });
  } catch (error) {
    console.error("Error updating category service:", error.message);
    res.status(500).json({ message: "Server error", error: error.message });
  }
};


export const saveCategoryServices = async (req, res) => {
  try {
    const { categoryId, services } = req.body;
    console.log("Payload:", categoryId, services);

    if (!categoryId || !services || !Array.isArray(services)) {
      return res.status(400).json({ message: "Category and services are required." });
    }

    const newService = new CategoryService({ categoryId, services });
    await newService.save();

    res.status(201).json({ message: "Category service saved successfully.", service: newService });
  } catch (error) {
    console.error("Error saving category service:", error);
    res.status(500).json({ message: "Server error", error });
  }
};




export const updateServiceTime = async (req, res) => {
  try {
    const { id: categoryId } = req.params; // categoryId will be passed in URL
    const updateData = req.body;

    const updated = await ServiceTime.findOneAndUpdate(
      { categoryId },        // 👈 find by categoryId
      updateData,
      { new: true }
    );

    if (!updated) {
      return res.status(404).json({ status: false, message: "Service time not found for this category" });
    }

    res.status(200).json({ status: true, message: "Updated successfully", data: updated });
  } catch (error) {
    console.error("Error updating service time:", error);
    res.status(500).json({ status: false, message: "Failed to update", error });
  }
};
