
import Location from '../models/Location.js';
import ProviderService from '../models/ProviderService.js';
import LocationProvider from '../models/LocationProvider.js';
import ServiceProvider from '../models/ServiceProvider.js';
import ProviderLocation from '../models/ProviderLocation.js';
import mongoose from 'mongoose';


export const createLocation = async (req, res) => {
  console.log(req.body);
  
  try {
    const {
      locationName,
      description,
      isVisible,
      isDefault,
      position,
      phone,
      address1,
      address2,
      city,
      zip,
      country
    } = req.body;

    const newLocation = new Location({
      locationName,
      description,
      image: req.file ? `/uploads/${req.file.filename}` : null,
      isVisible: isVisible === 'true',
      isDefault: isDefault === 'true',
      position,
      address: {
        phone,
        address1,
        address2,
        city,
        zip,
        country,
      }
    });

    await newLocation.save();
    res.status(201).json({ message: 'Location created successfully!', location: newLocation });
  } catch (error) {
    console.error('Error saving location:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};



export const updateLocation = async (req, res) => {
  try {
    const locationId = req.params.id;
    const {
      locationName,
      description,
      isVisible,
      isDefault,
      position,
      phone,
      address1,
      address2,
      city,
      zip,
      country
    } = req.body;

    // Find the location
    const location = await Location.findById(locationId);
    if (!location) {
      return res.status(404).json({ error: 'Location not found' });
    }

    // Update fields
    location.locationName = locationName;
    location.description = description;
    location.isVisible = isVisible === 'true' || isVisible === true;
    location.isDefault = isDefault === 'true' || isDefault === true;
    location.position = position;

    // Update image if a new one is uploaded
    if (req.file) {
      location.image = `/uploads/${req.file.filename}`;
    }

    // Update address
    location.address = {
      phone,
      address1,
      address2,
      city,
      zip,
      country
    };

    await location.save();

    res.status(200).json({ message: 'Location updated successfully!', location });

  } catch (error) {
    console.error('Error updating location:', error);
    res.status(500).json({ error: 'Internal server error' });
  }
};



export const getServiceProviders = async (req, res) => {
  try {
    
    const services = await ProviderService.find().populate('providerId');

    // const providers = await ServiceProvider.find();
    // const id=providers._id;
    // const existing = await ProviderService.findOne({ id });
    console.log("providers",services);
    
     
   
    res.status(200).json({ data: services });
  } catch (error) {
    console.error('Error fetching providers:', error);
    res.status(500).json({ error: 'Server error' });
  }
};


export const saveLocationProviders = async (req, res) => {
  try {
    const { locationId, providers } = req.body;

    // Validation
    if (!locationId || !mongoose.Types.ObjectId.isValid(locationId)) {
      return res.status(400).json({ error: 'Invalid or missing locationId' });
    }

    if (!Array.isArray(providers) || providers.some(p => !mongoose.Types.ObjectId.isValid(p))) {
      return res.status(400).json({ error: 'Invalid provider IDs' });
    }

    // Check if record exists
    let record = await LocationProvider.findOne({ locationId });

    if (record) {
      record.providers = providers;
      await record.save();
    } else {
      record = new LocationProvider({ locationId, providers });
      await record.save();
    }

    res.status(200).json({
      message: 'Location providers saved successfully',
      data: record,
    });
  } catch (error) {
    console.error('Error saving providers:', error);
    res.status(500).json({ error: 'Server error' });
  }
};

export const getAllLocations = async (req, res) => {
  try {
    const locations = await Location.find();

    const enrichedLocations = await Promise.all(
      locations.map(async (loc) => {
        // Get the provider info for this location
        const locProv = await LocationProvider.findOne({ locationId: loc._id });

        // const data =  await ProviderService.find().populate('providerId');

        let providerDetails = [];

        if (locProv && locProv.providers.length) {
          providerDetails = await ServiceProvider.find({
            _id: { $in: locProv.providers }
          });
        }

        return {
          ...loc._doc,
          providers: providerDetails
        };
      })
    );

    res.status(200).json({ status: true, data: enrichedLocations });
  } catch (error) {
    console.error('Error fetching locations:', error);
    res.status(500).json({ status: false, message: 'Internal server error' });
  }
};

export const getServiceProviderById = async (req, res) => {
  const { id } = req.params;

  // ✅ Validate ObjectId before querying
  if (!mongoose.Types.ObjectId.isValid(id)) {
    return res.status(400).json({ message: 'Invalid service provider ID' });
  }

  try {
    const service = await ServiceProvider.findById(id);
    if (!service) {
      return res.status(404).json({ message: 'Service not found' });
    }

    const serviceTime = await ProviderService.findOne({ providerId: id });

    const servicePageProviders = await ProviderLocation.findOne({ providerId: id })
      .populate('locationIds'); // Optionally populate related data

    res.status(200).json({
      service,
      serviceTime,
      servicePageProviders,
    });
  } catch (error) {
    console.error('Error fetching service by ID:', error);
    res.status(500).json({ message: 'Failed to fetch service' });
  }
};



// export const saveServiceProvider = async (req, res) => {
//   try {
//     const {
//       name,
//       description,
//       phone,
//       email,
//       clientsAtOnce,
//       showOnBooking,
//     } = req.body;

//     if (!name || !description) {
//       return res.status(400).json({ message: 'Name and description are required' });
//     }

//     const provider = new ServiceProvider({
//       name,
//       description,
//       phone,
//       email,
//       clientsAtOnce,
//       showOnBooking,
//     });

//     await provider.save();

//     res.status(200).json({ message: 'Provider saved successfully', provider });
//   } catch (error) {
//     console.error('Error saving provider:', error);
//     res.status(500).json({ error: 'Server error' });
//   }
// };



export const deleteLocation = async (req, res) => {
  try {
    const { id } = req.params;
    await Location.findByIdAndDelete(id);
    res.status(200).json({ success: true, message: "Location deleted." });
  } catch (err) {
    res.status(500).json({ success: false, message: "Delete failed", error: err.message });
  }
};




export const updateLocationProviders = async (req, res) => {
  try {
    const { locationId } = req.params;
    const { providers } = req.body;

    // if (!locationId || !mongoose.Types.ObjectId.isValid(locationId)) {
    //   return res.status(400).json({ error: 'Invalid or missing locationId' });
    // }

    const locationObjectId = new mongoose.Types.ObjectId(locationId);

    if (!Array.isArray(providers) || providers.some(p => !mongoose.Types.ObjectId.isValid(p))) {
      return res.status(400).json({ error: 'Invalid provider IDs' });
    }

    const providerObjectIds = providers.map(id => new mongoose.Types.ObjectId(id));

    // ✅ Make sure you use the correct field (location or locationId)
    const updatedRecord = await LocationProvider.findOneAndUpdate(
      { locationId: locationObjectId }, // <-- Use correct key
      { providers: providerObjectIds },
      { new: true }
    );

    if (!updatedRecord) {
      return res.status(404).json({ error: 'Location provider record not found' });
    }

    res.status(200).json({
      message: 'Location providers updated successfully',
      data: updatedRecord,
    });
  } catch (error) {
    console.error('Error updating location providers:', error);
    res.status(500).json({ error: 'Server error' });
  }
};
