import Package from "../models/Package.js";
import PackageService from "../models/PackageService.js";
import IssuedPackage from "../models/IssuedPackage.js";
import IssuePackageAssignedToClient from "../models/IssuePackageAssignedToClient.js";
import mongoose from "mongoose";

export const createPackage = async (req, res) => {
  console.log("checks", req.body);

  try {
    const { filename } = req.file || {};

    // ✅ Parse duration if it's a string
    if (req.body.duration && typeof req.body.duration === "string") {
      try {
        req.body.duration = JSON.parse(req.body.duration);
      } catch (err) {
        console.error("Invalid duration format:", err);
        return res.status(400).json({ success: false, message: "Invalid duration format" });
      }
    }

    const data = {
      ...req.body,
      image: filename || "",
    };

    const membership = new Package(data);
    await membership.save();

    res.status(201).json({ success: true, membership });
  } catch (error) {
    res.status(500).json({ success: false, message: error.message });
  }
};


export const getAllPackage = async (req, res) => {
  try {
    const memberships = await Package.find().sort({ createdAt: -1 });
    res.status(200).json({ success: true, memberships });
  } catch (error) {
    res.status(500).json({ success: false, message: error.message });
  }
};


export const getPackageById = async (req, res) => {
  try {
    const { id } = req.params;
    const packageData = await Package.findById(id);

    if (!packageData) {
      return res.status(404).json({ success: false, message: "Package not found" });
    }

    res.status(200).json({ success: true, package: packageData });
  } catch (error) {
    res.status(500).json({ success: false, message: error.message });
  }
};


export const deletePackageById = async (req, res) => {
  try {
    const { id } = req.params;

    const deletedPackage = await Package.findByIdAndDelete(id);

    if (!deletedPackage) {
      return res.status(404).json({ success: false, message: "Package not found" });
    }

    res.status(200).json({ success: true, message: "Package deleted successfully" });
  } catch (error) {
    res.status(500).json({ success: false, message: error.message });
  }
};



export const savePackageServices = async (req, res) => {
  const { packageId, services } = req.body;

  try {
    let existing = await PackageService.findOne({ packageId });

    if (existing) {
      existing.services = services;
      await existing.save();
      return res.status(200).json({ success: true, message: "Updated successfully", data: existing });
    } else {
      const newEntry = new PackageService({ packageId, services });
      await newEntry.save();
      return res.status(201).json({ success: true, message: "Saved successfully", data: newEntry });
    }
  } catch (err) {
    console.error("Error saving package services:", err);
    res.status(500).json({ success: false, message: "Server error" });
  }
};

// Get services for a specific package
export const getPackageServices = async (req, res) => {
  const { id } = req.params;

  try {
    const packageService = await PackageService.findOne({ packageId: id }).populate("services", "name _id");

    if (!packageService) {
      return res.status(200).json({ success: true, services: [] });
    }

    const serviceIds = packageService.services.map((s) => s._id);
    res.status(200).json({ success: true, services: serviceIds });
  } catch (err) {
    console.error("Error fetching package services:", err);
    res.status(500).json({ success: false, message: "Server error" });
  }
};






export const issuePackage = async (req, res) => {
  try {
    const { packageId, serviceName, duration, startDate } = req.body;

    if (!packageId || !serviceName || !duration || !startDate) {
      return res.status(400).json({
        success: false,
        message: "Missing required fields",
      });
    }

    const issuedPackage = await IssuedPackage.create({
      packageId,
      serviceName,
      duration,
      startDate,
    });

    res.status(201).json({ success: true, issuedPackage });
  } catch (error) {
    res.status(500).json({ success: false, message: error.message });
  }
};



export const assignPackageToClients = async (req, res) => {
  const { issuePackageId, packageId, clients } = req.body;

  if (!issuePackageId || !packageId || !clients?.length) {
    return res.status(400).json({ error: "Missing required fields" });
  }

  try {
    const newAssignment = await IssuePackageAssignedToClient.create({
      issuePackageId,
      packageId,
      assignedClients: clients,
    });

    res.status(201).json({
      message: "Package assigned successfully",
      data: newAssignment,
    });
  } catch (error) {
    console.error("Assignment error:", error);
    res.status(500).json({ error: "Failed to assign package" });
  }
};



export const updatePackageById = async (req, res) => {
  try {
    const { id } = req.params;

    // Optional: console.log(req.body) if not using multipart
    // console.log(req.body);

    const updatedFields = {
      serviceName: req.body.serviceName,
      description: req.body.description,
      price: req.body.price,
      isRecurring: req.body.isRecurring,
      duration: JSON.parse(req.body.duration), // if sent as stringified JSON
      Tax: req.body.salesTax,
      DurationPeriod:req.body.DurationPeriod,
      enableLimit: req.body.enableLimit === 'true',
      packageLimit: req.body.packageLimit,
      maxPackages: req.body.maxPackages,
      unlimited: req.body.unlimited === 'true',
    
    };

    // Handle image upload
    if (req.file) {
      updatedFields.image = req.file.filename;
    }

    const updatedPackage = await Package.findByIdAndUpdate(id, updatedFields, {
      new: true,
      runValidators: true,
    });

    if (!updatedPackage) {
      return res.status(404).json({ success: false, message: "Package not found" });
    }

    res.status(200).json({ success: true, message: "Package updated successfully", package: updatedPackage });
  } catch (error) {
    console.error(error);
    res.status(500).json({ success: false, message: error.message });
  }
};


export const getIssuedPackage = async (req, res) => {
  try {
    const { id } = req.params;
    console.log("ids",id);
    

    const issuedPackage = await IssuedPackage.findOne({ packageId: id });

    if (!issuedPackage) {
      return res.status(200).json({
        success: true,
        issuedPackage: null, // indicate not issued yet
      });
    }

    res.status(200).json({
      success: true,
      issuedPackage,
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: "Error fetching issued package",
    });
  }
};


export const updateIssuedPackage = async (req, res) => {
  try {
    const { id } = req.params;
    const { serviceName, duration, startDate } = req.body;

    // Validate required fields
    if (!startDate || !duration || !serviceName) {
      return res.status(400).json({
        success: false,
        message: "Missing required fields",
      });
    }

    const updated = await IssuedPackage.findByIdAndUpdate(
      id,
      {
        serviceName,
        duration,
        startDate,
      },
      { new: true }
    );

    if (!updated) {
      return res.status(404).json({
        success: false,
        message: "Issued package not found",
      });
    }

    res.status(200).json({
      success: true,
      message: "Issued package updated successfully",
      updatedPackage: updated,
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: "Failed to update issued package",
    });
  }
};



export const getAssignedPackageClients = async (req, res) => {
 
  
  const { packageId } = req.params;
   console.log("yyyyy",packageId);

  if (!packageId) {
    return res.status(400).json({ error: "Missing issuePackageId or packageId" });
  }

  try {
    const assignment = await IssuePackageAssignedToClient.findOne({
      packageId,
    });

    if (!assignment) {
      return res.status(200).json({
        message: "No assignment found",
        assignedClients: [],
      });
    }

    res.status(200).json({
      message: "Assignment fetched successfully",
      assignedClients: assignment.assignedClients,
    });
  } catch (error) {
    console.error("Fetch error:", error);
    res.status(500).json({ error: "Failed to fetch assigned clients" });
  }
};


export const updateAssignedPackageClients = async (req, res) => {
  const { packageId } = req.params;
  const { clients } = req.body;

  if (!packageId || !Array.isArray(clients) || clients.length === 0) {
    return res.status(400).json({ error: "Missing packageId or clients array" });
  }

  try {
    // Find existing assignment by packageId
    const existingAssignment = await IssuePackageAssignedToClient.findOne({ packageId });

    if (!existingAssignment) {
      return res.status(404).json({ error: "No existing assignment found for this package" });
    }

    // Update assignedClients
    existingAssignment.assignedClients = clients;
    await existingAssignment.save();

    res.status(200).json({
      message: "Assigned clients updated successfully",
      assignedClients: existingAssignment.assignedClients,
    });
  } catch (error) {
    console.error("Update error:", error);
    res.status(500).json({ error: "Failed to update assigned clients" });
  }
};


export const deleteIssuedPackage = async (req, res) => {
  
  
  try {
    const { id } = req.params;
    console.log("not",id);

    if (!id) {
      return res.status(400).json({
        success: false,
        message: "Missing issued package ID",
      });
    }

    const objectId = new mongoose.Types.ObjectId(id);

    const deleted = await IssuedPackage.findByIdAndDelete(objectId);

    if (!deleted) {
      return res.status(404).json({
        success: false,
        message: "Issued package not found",
      });
    }

    res.status(200).json({
      success: true,
      message: "Issued package deleted successfully",
      deletedPackage: deleted,
    });
  } catch (error) {
    console.error("Delete error:", error);
    res.status(500).json({
      success: false,
      message: "Failed to delete issued package",
    });
  }
};
