// controllers/CoverTemplateController.js
const path = require("path");
const CoverTemplate = require("../models/covertemplate");

// build absolute URL from a stored path like "/api/templete/rose.png"
const toAbsolute = (p, req) => {
  if (!p) return null;
  if (/^https?:\/\//i.test(p)) return p; // already absolute
  const clean = p.startsWith("/") ? p : `/${p}`;
  return `${req.protocol}://${req.get("host")}${clean}`;
};

/**
 * GET /api/cover-templates
 */
exports.list = async (req, res) => {
  try {
    const rows = await CoverTemplate.findAll({
      where: { is_active: true },
      order: [["id", "ASC"]],
      attributes: ["id", "title", "file_path", "preview_path"],
    });

    const data = rows.map((r) => ({
      id: r.id,
      name: r.title,
      assetUrl: toAbsolute(r.file_path, req),
      previewUrl: toAbsolute(r.preview_path || r.file_path, req),
    }));

    return res.json(data);
  } catch (e) {
    console.error(e);
    return res.status(500).json({ message: "Failed to load templates" });
  }
};

/**
 * POST /api/cover-templates
 * Files: asset (required), preview (optional)
 */
exports.create = async (req, res) => {
  try {
    const { title, description } = req.body;
    if (!title) return res.status(400).json({ message: "Title is required" });

    const assetFile = req.files?.asset?.[0];
    const previewFile = req.files?.preview?.[0];
    if (!assetFile) {
      return res.status(400).json({ message: "Asset image is required" });
    }

    const assetFileName = path.basename(assetFile.path);
    const previewFileName = previewFile ? path.basename(previewFile.path) : null;

    // store relative paths in DB
    const assetRel = `/api/templete/${assetFileName}`;
    const previewRel = previewFileName ? `/api/templete/${previewFileName}` : assetRel;

    const created = await CoverTemplate.create({
      title,
      description: description || null,
      file_path: assetRel,
      preview_path: previewRel,
      is_active: true,
    });

    // return absolute URLs to client
    return res.status(201).json({
      id: created.id,
      name: created.title,
      assetUrl: toAbsolute(created.file_path, req),
      previewUrl: toAbsolute(created.preview_path || created.file_path, req),
      message: "Template created",
    });
  } catch (e) {
    console.error(e);
    return res.status(500).json({ message: "Failed to create template" });
  }
};

/**
 * PATCH /api/cover-templates/:id
 */
exports.update = async (req, res) => {
  try {
    const { id } = req.params;
    const { title, description, is_active } = req.body;

    const row = await CoverTemplate.findByPk(id);
    if (!row) return res.status(404).json({ message: "Template not found" });

    if (typeof title !== "undefined") row.title = title;
    if (typeof description !== "undefined") row.description = description;
    if (typeof is_active !== "undefined") row.is_active = !!is_active;

    await row.save();

    return res.json({
      id: row.id,
      name: row.title,
      assetUrl: toAbsolute(row.file_path, req),
      previewUrl: toAbsolute(row.preview_path || row.file_path, req),
      is_active: row.is_active,
      message: "Template updated",
    });
  } catch (e) {
    console.error(e);
    return res.status(500).json({ message: "Failed to update template" });
  }
};

/**
 * DELETE /api/cover-templates/:id  (soft delete)
 */
exports.remove = async (req, res) => {
  try {
    const { id } = req.params;
    const row = await CoverTemplate.findByPk(id);
    if (!row) return res.status(404).json({ message: "Template not found" });
    row.is_active = false;
    await row.save();
    return res.json({ message: "Template deactivated" });
  } catch (e) {
    console.error(e);
    return res.status(500).json({ message: "Failed to remove template" });
  }
};
