const { Op } = require("sequelize");
const Anniversary = require("../models/Anniversary");
const Memorial = require("../models/Memorial");
const User = require("../models/User");
const CoAdmins = require("../models/CoAdmins");
const Comment = require("../models/Comment");

module.exports = {
  async create(req, res) {
    try {
      const anniversary = await Anniversary.create({ ...req.body });
      return res.json({ message: "Created new anniversary", anniversary });
    } catch (err) {
      console.log({ err });
    }
  },

  async index(req, res) {
    try {
      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      const anniversaries = await Anniversary.findAndCountAll({
        where: {
          ...(req.query.search && {
            [Op.or]: [
              { title: { [Op.regexp]: req.query.search } },
              { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
              { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
              { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
            ],
          }),
        },
        order: [["createdAt", "DESC"]],
        limit: parseInt(pageSize),
        offset: (parseInt(page) - 1) * parseInt(pageSize),
        include: [
          { model: Memorial, as: "memorial" },
          { model: User, as: "user" },
        ],
      });
      const total = Math.ceil(anniversaries.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        anniversaries: anniversaries.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

  async getByMemorial(req, res) {
    try {
      const anniversaries = await Anniversary.findAll({
        where: { memorial_id: req.params.id },
        include: [{ model: User, as: "user" }],
      });
      return res.json(anniversaries);
    } catch (err) {
      console.log({ err });
    }
  },

  async getAnniversariesByCoAdmins(req, res) {
    try {
      // Step 1: Get all memorials where the user is a co-admin
      const coAdminMemorials = await CoAdmins.findAll({
        where: { user_id: req.params.userId },
        attributes: ["memorial_id"],
      });
      // Extract memorial IDs from the result
      const memorialIds = coAdminMemorials.map(
        (coAdmin) => coAdmin.memorial_id
      );
      console.log({ memorialIds });

      if (memorialIds.length === 0) {
        return []; // No memorials found for this co-admin
      }

      const page = req.query.page ? req.query.page : 1;
      const pageSize = req.query.pageSize ? req.query.pageSize : 4;
      // Step 2: Fetch anniversaries linked to these memorial IDs
      const anniversaries = await Anniversary.findAndCountAll({
        where: { memorial_id: memorialIds },
        ...(req.query.search && {
          [Op.or]: [
            { title: { [Op.regexp]: req.query.search } },
            { "$Memorial.Firstname$": { [Op.regexp]: req.query.search } },
            { "$Memorial.MiddleName$": { [Op.regexp]: req.query.search } },
            { "$Memorial.LastName$": { [Op.regexp]: req.query.search } },
          ],
        }),
        include: [
          {
            model: Memorial,
            as: "memorial",
          },
        ],
      });
      const total = Math.ceil(anniversaries.count / pageSize);
      return res.status(201).json({
        filters: { page: parseInt(page), nextPage: parseInt(page) + 1, total },
        anniversaries: anniversaries.rows,
      });
    } catch (err) {
      console.log({ err });
    }
  },

  async getSingleAnniversary(req, res) {
    try {
      const anniversary = await Anniversary.findOne({
        where: { id: req.params.id },
        include: [
          { model: User, as: "user" },
          { model: Memorial, as: "memorial" },
        ],
      });
      return res.json(anniversary);
    } catch (err) {
      console.log({ err });
    }
  },

  async update(req, res) {
    try {
      const anniversary = await Anniversary.update(
        { ...req.body },
        { where: { id: req.params.id } }
      );
      return res.json({ message: "Updated anniversary", anniversary });
    } catch (err) {
      console.log({ err });
    }
  },
  async remove(req, res) {
    try {
      await Anniversary.destroy({
        where: { id: req.params.id },
      });
      return res.json({ message: "Removed new anniversary" });
    } catch (err) {
      console.log({ err });
    }
  },

  async AddComment(req, res) {
    const { anniversary_id, user_id, comment } = req.body;

    if (!comment || comment.trim() === "")
      return res.status(400).json({ error: "Comment cannot be empty" });

    if (comment.trim().split(/\s+/).length > 250)
      return res.status(400).json({ error: "Comment cannot exceed 250 words" });

    try {
      const newComment = await Comment.create({ anniversary_id, user_id, comment });
      res.status(200).json({ message: "Comment added", comment: newComment });
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: "Server error" });
    }
  }
  ,
  async GetComments(req, res) {
    const { anniversary_id } = req.params;
    console.log("getcomment", anniversary_id);

    try {
      const comments = await Comment.findAll({
        where: { anniversary_id },
        include: [
          {
            model: User,
            attributes: ["userid", "firstName", "lastName"],
          },
        ],
      });

      res.status(200).json(comments);
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: "Server error" });
    }
  }

  ,
  // Update Comment
  async UpdateComment(req, res) {
    const { id } = req.params;
    const { comment } = req.body;

    console.log("idssssssssssssssssss", id)

    if (!comment || comment.trim() === "")
      return res.status(400).json({ error: "Comment cannot be empty" });

    if (comment.trim().split(/\s+/).length > 250)
      return res.status(400).json({ error: "Comment cannot exceed 250 words" });

    try {
      const updated = await Comment.update(
        { comment },
        { where: { id } }
      );

      if (updated[0] === 0) {
        return res.status(404).json({ error: "Comment not found" });
      }

      res.status(200).json({ message: "Comment updated" });
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: "Server error" });
    }
  }
  ,
  // Delete Comment
  async DeleteComment(req, res) {
    const { id } = req.params;

    try {
      const deleted = await Comment.destroy({ where: { id } });

      if (deleted === 0) {
        return res.status(404).json({ error: "Comment not found" });
      }

      res.status(200).json({ message: "Comment deleted" });
    } catch (err) {
      console.error(err);
      res.status(500).json({ error: "Server error" });
    }
  },
 async getAnniversariesByUser(req, res) {
  try {
    const { userId } = req.params;

    const page = req.query.page ? parseInt(req.query.page, 10) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize, 10) : 4;

    const anniversaries = await Anniversary.findAndCountAll({
      where: {
        user_id: userId,
        ...(req.query.search && {
          [Op.or]: [
            { title: { [Op.regexp]: req.query.search } },
            { "$memorial.Firstname$": { [Op.regexp]: req.query.search } },
            { "$memorial.MiddleName$": { [Op.regexp]: req.query.search } },
            { "$memorial.LastName$": { [Op.regexp]: req.query.search } },
          ],
        }),
      },
      include: [
        {
          model: User,
          as: "user",
          attributes: ["userid", "firstName", "lastName", "email"],
        },
        { model: Memorial, as: "memorial" },
      ],
      order: [["createdAt", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
    });

    const totalPages = Math.ceil(anniversaries.count / pageSize);

    return res.status(200).json({
      filters: { page, nextPage: page + 1, total: totalPages },
      anniversaries: anniversaries.rows,
    });
  } catch (error) {
    console.error("Error fetching anniversaries by user:", error);
    return res.status(500).json({ error: "Internal Server Error" });
  }
}

};
