// controller/anniversary.controller.js
const { Op } = require("sequelize");
const Anniversary = require("../models/Anniversary");
const User = require("../models/User");
const Memorial = require("../models/Memorial");

// ✅ Get all anniversaries (Super Admin only)
exports.getAllAnniversaries = async (req, res) => {
  try {
    const page = req.query.page ? parseInt(req.query.page) : 1;
    const pageSize = req.query.pageSize ? parseInt(req.query.pageSize) : 10;

    const anniversaries = await Anniversary.findAndCountAll({
      where: {
        ...(req.query.search && {
          [Op.or]: [
            { title: { [Op.like]: `%${req.query.search}%` } },
            { description: { [Op.like]: `%${req.query.search}%` } },
            { "$user.firstName$": { [Op.like]: `%${req.query.search}%` } },
            { "$user.lastName$": { [Op.like]: `%${req.query.search}%` } },
            { "$memorial.title$": { [Op.like]: `%${req.query.search}%` } },
          ],
        }),
      },
      include: [
        { model: User, as: "user", attributes: ["userid", "firstName", "lastName", "email"] },
        { model: Memorial, as: "memorial", attributes: ["id", "title"] },
      ],
      order: [["createdAt", "DESC"]],
      limit: pageSize,
      offset: (page - 1) * pageSize,
    });

    const total = Math.ceil(anniversaries.count / pageSize);

    return res.json({
      filters: { page, nextPage: page + 1, total },
      anniversaries: anniversaries.rows,
    });
  } catch (error) {
    console.error("getAllAnniversaries error:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

// ✅ Get single anniversary
exports.getAnniversary = async (req, res) => {
  try {
    const anniversary = await Anniversary.findByPk(req.params.id, {
      include: [
        { model: User, as: "user", attributes: ["userid", "firstName", "lastName", "email"] },
        { model: Memorial, as: "memorial", attributes: ["id", "title"] },
      ],
    });
    if (!anniversary) return res.status(404).json({ message: "Not found" });
    res.json(anniversary);
  } catch (err) {
    console.error("getAnniversary error:", err);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

// ✅ Update anniversary
exports.updateAnniversary = async (req, res) => {
  try {
    const { title, description } = req.body;
    const anniversary = await Anniversary.findByPk(req.params.id);
    if (!anniversary) return res.status(404).json({ message: "Not found" });

    anniversary.title = title || anniversary.title;
    anniversary.description = description || anniversary.description;
    await anniversary.save();

    res.json({ message: "Anniversary updated successfully", anniversary });
  } catch (err) {
    console.error("updateAnniversary error:", err);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

// ✅ Delete anniversary
exports.deleteAnniversary = async (req, res) => {
  try {
    const anniversary = await Anniversary.findByPk(req.params.id);
    if (!anniversary) return res.status(404).json({ message: "Not found" });

    await anniversary.destroy();
    res.json({ message: "Anniversary deleted successfully" });
  } catch (err) {
    console.error("deleteAnniversary error:", err);
    res.status(500).json({ error: "Internal Server Error" });
  }
};
