const { HelpThread, HelpMessage, HelpThreadParticipant } = require("../models/index");
const User = require("../models/User");
const Notification = require("../models/Notification");
const CoAdmins = require("../models/CoAdmins");
const Memorial = require("../models/Memorial");
const Obituary = require("../models/Obituary");

const isSuperAdmin = (me) => me?.type === "super_admin";
const isServiceProvider = (me) => me?.type === "service_provider";

// Helpers to find audience groups
async function findMemorialAdminIds() {
  const owners = await Memorial.findAll({ attributes: ["user_id"], raw: true });
  const coadmins = await CoAdmins.findAll({ attributes: ["user_id"], raw: true });
  const ids = new Set([...owners.map(o => o.user_id), ...coadmins.map(c => c.user_id)]);
  return Array.from(ids);
}
async function findObituaryAdminIds() {
  const owners = await Obituary.findAll({ attributes: ["user_id"], raw: true });
  return Array.from(new Set(owners.map(o => o.user_id)));
}

/**
 * POST /help-center/threads
 */
exports.createThread = async (req, res) => {
  try {
    const me = req.user;
    const { subject, body, scope = "custom", recipientIds = [] } = req.body;

    if (!me) return res.status(401).json({ error: "Unauthorized" });
    if (!subject || !body) return res.status(400).json({ error: "subject and body are required" });

    let targets = [];
    if (scope === "memorial_admins") {
      targets = await findMemorialAdminIds();
    } else if (scope === "obituary_admins") {
      targets = await findObituaryAdminIds();
    } else if (scope === "service_providers") {
      targets = (
        await User.findAll({ where: { type: "service_provider" }, attributes: ["userid"], raw: true })
      ).map(u => u.userid);
    } else {
      targets = recipientIds;
    }

    if (isServiceProvider(me)) {
      const superAdmins = (
        await User.findAll({ where: { type: "super_admin" }, attributes: ["userid"], raw: true })
      ).map(u => u.userid);
      targets = targets.filter(id => superAdmins.includes(id));
    }

    const uniqueTargets = Array.from(new Set(targets)).filter(id => id !== me.userid);

    const thread = await HelpThread.create({ subject, created_by: me.userid, scope });

    await HelpThreadParticipant.bulkCreate(
      uniqueTargets.concat(me.userid).map(uid => ({ thread_id: thread.id, user_id: uid }))
    );

    const msg = await HelpMessage.create({
      thread_id: thread.id,
      sender_id: me.userid,
      body,
    });

    // ✅ FIX: Notification type = help_center
    await Notification.bulkCreate(
      uniqueTargets.map(uid => ({
        text: `New Help Center message: ${subject}`,
        user_id: uid,
        type: "help_center",
        read: false,
      }))
    );

    return res.json({ threadId: thread.id, messageId: msg.id });
  } catch (e) {
    console.error(e);
    return res.status(500).json({ error: "Failed to create thread" });
  }
};

/**
 * GET /help-center/threads
 */
exports.listMyThreads = async (req, res) => {
  try {
    const me = req.user;
    if (!me) return res.status(401).json({ error: "Unauthorized" });

    const threads = await HelpThread.findAll({
      include: [
        { association: "participants", where: { user_id: me.userid }, required: true },
        { association: "messages", include: ["sender"], order: [["createdAt", "DESC"]] },
        { association: "creator", attributes: ["userid", "firstName", "lastName"] },
      ],
      order: [["updatedAt", "DESC"]],
    });

    return res.json(threads);
  } catch (e) {
    console.error(e);
    return res.status(500).json({ error: "Failed to fetch threads" });
  }
};

/**
 * GET /help-center/threads/:id
 */
exports.getThread = async (req, res) => {
  try {
    const me = req.user;
    const { id } = req.params;
    if (!me) return res.status(401).json({ error: "Unauthorized" });

    const thread = await HelpThread.findByPk(id, {
      include: [
        {
          association: "participants",
          include: [{ association: "user", attributes: ["userid", "firstName", "lastName"] }],
        },
        {
          association: "messages",
          include: [{ association: "sender", attributes: ["userid", "firstName", "lastName"] }],
          order: [["createdAt", "ASC"]],
        },
      ],
    });

    if (!thread) return res.status(404).json({ error: "Not found" });
    if (!thread.participants.some(p => p.user_id === me.userid)) {
      return res.status(403).json({ error: "Forbidden" });
    }

    return res.json(thread);
  } catch (e) {
    console.error(e);
    return res.status(500).json({ error: "Failed to fetch thread" });
  }
};

/**
 * POST /help-center/threads/:id/messages
 */
exports.replyToThread = async (req, res) => {
  try {
    const me = req.user;
    const { id } = req.params;
    const { body } = req.body;

    if (!me) return res.status(401).json({ error: "Unauthorized" });
    if (!body) return res.status(400).json({ error: "body is required" });

    const isParticipant = await HelpThreadParticipant.findOne({
      where: { thread_id: id, user_id: me.userid },
      raw: true,
    });
    if (!isParticipant) return res.status(403).json({ error: "Forbidden" });

    const message = await HelpMessage.create({
      thread_id: id,
      sender_id: me.userid,
      body,
    });

    const participants = await HelpThreadParticipant.findAll({
      where: { thread_id: id },
      attributes: ["user_id"],
      raw: true,
    });
    const others = participants.map(p => p.user_id).filter(uid => uid !== me.userid);

    // ✅ FIX: Notification type = help_center
    await Notification.bulkCreate(
      others.map(uid => ({
        text: "New reply in Help Center thread",
        user_id: uid,
        type: "help_center",
        read: false,
      }))
    );

    return res.json({ messageId: message.id });
  } catch (e) {
    console.error(e);
    return res.status(500).json({ error: "Failed to post reply" });
  }
};

/**
 * GET /help-center/unread-count
 */
exports.unreadCount = async (req, res) => {
  try {
    const me = req.user;
    console.log(me);
    
    if (!me) return res.status(401).json({ error: "Unauthorized" });

    const count = await Notification.count({
      where: {
        user_id: me.userid,
        read: false,
        type: "help_center", 
      },
    });

    return res.json({ unread: count });
  } catch (err) {
    console.error("unreadCount error:", err);
    return res.json({ unread: 0 });
  }
};
